'**********************************************************************************
'*                   A DeltaCad macro with a number of functions                  *
'*                        for use in Sundial programs                             * 
'*      Copyright 2006 Fer de Vries & The North American Sundial Society          * 
'*                            http://sundials.org                                 *
'*                                                                                *
'*   These functions are used for a program to calculate a azimuthal nomograph.   *
'*                      Idea for the nomograph by Mac Oglesby, USA.               *
'*                               Name AzNomograph.bas                             *
'*                                                                                *
'*      This macro may be circulated and modified as long as this header          *
'*                              remains intact.                                   *
'*                                                                                *
'**********************************************************************************

'date november 2006

'In february 2007 this macro is extended with Italian and Babylonian hourlines.
'The Italian hourlines are counted as "hours to sunset".
'These hourlines are in separate layers  It-...  and  Bab-....
'Fer de Vries.

'**********************************************************************************


'In this Macro formulas dsin, dcos and dtan and reverse functions use degrees.


'MAIN FUNCTIONS AND SUBS IN THIS MACRO :

'dSin(x),  dCos(x)   and dTan(x)   : x is in degrees 
'dAsin(x), dAcos(x), and dAtan(x)  : returns x in degrees

'CalcYearFormula(InitYear)
' --- calculates formulas for EoT and decl. for given year in YearFormula(13)

'Function CalcJulDay(ByVal xyear,xmonth,xday As integer, ByVal xhour As double) As Double
' --- Calculates Julian day out of year,month,day,hour in Gregorian calendar.
'This function is used by CalcYearFormula(InitYear)

'Function EoT(ByVal daynr As double) As double
' --- Calculates Eot out of daynumber
'With variable fact in the procedure output may be choosen to be in
'seconds of time, minutes of time, degrees or radians.

'Function decl(ByVal daynr As double) As Double
' --- Calculates sun's declination out of daynumber
'With variable fact in the procedure output may be choosen to be in
'degrees or radians.

'Function daynr(ByVal y,m,d,h,mi As double) As Double
' --- Calculates daynumber out of year, month, day, hour minutes

'Function halfdayarc(ByVal lat, ByVal decl As double) As double
' --- Calculates half day arc out of Latitude and declination
'With variable fact in the procedure output may be choosen to be in
'degrees or radians.


Option Explicit ' Force all variables to be declared before they are used. Limit adhoc variables.
'However, I used some in some functions. 

Dim pi, d2r, r2d As Double
Dim InitYear As Integer
Dim YearFormula(13) As Double

Dim action,button As String 'For input data
Dim outtext As String
Dim Latitude, LocalMeridian, StandardMeridian, declination, hourangle, LongCorr, EqOfTime As Double
Dim Azimuth, Hour, HalfDayLength As Double

Dim datetext(12) As String
Dim NumDaysInMonth(12) As Integer
 
Dim EoTfact, MonthFact, flag As Integer

Dim Scx, Scy, ScaleWidth, ScaleHeight, scalex, scaley, Xbox, Ybox, tickmark1, tickmark2, ScaleFont As Double
Dim x,y, x1,y1, x2,y2, deltax, deltay, TextX, TextY As Double

Dim count, count1, count2, count3, sum1, sum2, temp1, temp2, ChoiceBabIt As Integer

Dim Lijn(500) As Double

Dim index, index1, index2, linenumber As Integer

Dim hemi As Integer  'Northern hemisphere hemi = -1, Southern hemispher hemi = 1

'Maximize the window, close any existing drawing without saving, and start a new drawing.
dcSetDrawingWindowMode dcMaximizeWin
dcCloseWithoutSaving
dcNew ""

'Define pi, d2r, r2d
pi = 4*Atn(1)  'Calculates pi
d2r = pi/180   'Factor to convert degrees to radians
r2d = 180/pi   'Factor to convert radians to degrees

'Establish the 5 standard line thicknesses in thousandths of an inch.
dcSetDrawingData dcLineThin,   .003
dcSetDrawingData dcLineNormal, .008
dcSetDrawingData dcLineThick,  .012
dcSetDrawingData dcLineHeavy,  .024
dcSetDrawingData dcLineWide,   .048

'Set some parameters
dcSetLineParms dcBLACK, dcSOLID, dcTHIN
dcSetCircleParms dcBLACK, dcSOLID, dcTHIN
dcSetViewScale(1.75)  'Sets the dawingscale on screen to 1.75.
dcSetDrawingData dcScaleWithViewScale = true
'******************************************
'START OF PROGRAM

'If calculations of EoT and/or Declination of Sun are needed always start with 
'choosing or inputting InitYear and calculate YearFormula.
'In this program a init value (2010) is choosen. Change it if you like.

InitYear = 2010                      '***********
CalcYearFormula(InitYear)            '***********

'Place below the program as you want.'***********

'Initiate a number of values;
'Input Data for: 
  'Latitude, local meridian, zone meridian (Latitude between 23.44 to 66.56 and -23.44 to -66.56.)
  'year for calculation sun's declination and EoT. In this program an init value is choosen.
  'with or without EoT correction.
  '1 jan. or 1 jul. at top of nomogram

'Draw azimuth-date scale with text. For Southern Latitude different azimuth scale.
'Draw hour lines.
'DrawSunriseSunsetLines
'DrawItBabHourLines
'Add numbers for hours.

InitConstants
InputData
SetLayers               'Draw different parts in different layers.
DrawAzimuthDatescale    'Depends on northern or southern Latitude.
                        'Layer AzDateScale.
                        'Draws also Grid for datescale in layers Grid1 and Grid2.

DrawHourLines           'Layers Normal-Full-Hours, Normal-Half-Hours, Normal-Quarter-Hours.

DrawSunriseSunsetLines  'Layer SunRiseSet.

DrawItBabHourLines      'Layers It-Full-Hours, It-Half-Hours, It-Quarter-Hours,
                        'Bab-Full-Hours, Bab-Half-Hours, Bab-Quarter-Hours.

AddConstants            'Latitude, Longitude, Zone meridian.
DrawBoxAroundDrawing    'Layer default


dcSetCurrentLayer "default"
'dcviewall               'Show drawing in window on screen.

dcTurnLayerOff "It-Full-Hours"
dcTurnLayerOff "It-Half-Hours"
dcTurnLayerOff "It-Quarter-Hours"
dcTurnLayerOff "Bab-Full-Hours"
dcTurnLayerOff "Bab-Half-Hours"
dcTurnLayerOff "Bab-Quarter-Hours"


'END OF PROGRAM
'******************************************



'******************************************
'The Functions and Sub's

' --- dTan operates on degrees
Function dTan(ByVal value As double) As Double
dTan = Tan(value*d2r)
If Abs(dTan) < 1e-12 Then dTan = 0
End Function

' --- dAtan returns degrees
Function dAtan(ByVal value As double) As Double
dAtan = Atn(value)*r2d
End Function

' --- dSin operates on degrees
Function dSin(ByVal value As double) As Double
dSin = Sin(value * d2r)
If Abs(dSin) < 1e-12 Then dSin = 0
End Function

' --- dAsin returns degrees
Function dAsin(ByVal value As Double) As Double
If Abs(value) > 0.999999999999 Then
dAsin = 90 * sgn(value)
Else
dAsin = dAtan(value/Sqr(1-value*value))
End If
End Function

' --- dCos operates on degrees
Function dCos(ByVal value As double) As Double
dCos = Cos(value * d2r)
If Abs(dCos) < 1e-12 Then dCos = 0
End Function

' --- dAcos returns degrees
Function dAcos(ByVal value As Double) As Double
dAcos = 90 - dAsin(value)
End Function



' --- Calculates Julian day out of year,month,day,hour in Gregorian calendar.
Function CalcJulDay(ByVal xyear,xmonth,xday As integer, ByVal xhour As double) As Double

  Dim help1,help2 As Double
  
  If (xmonth=1) Or (xmonth=2) Then
    xmonth=xmonth+12
    xyear=xyear-1
  End If
  help1=Int(xyear/100)
  help2=2-help1+Int(help1/4)
  CalcJulDay=Int(365.25*xyear)+Int(30.6001*(xmonth+1))+xday+xhour/24+1720994.5+help2
End Function


' --- Calculates a formula for a year to calculate EoT and decl out of daynumber
Sub CalcYearFormula(ByVal xyear As Integer)

  Dim l,w,e,epsilon,d,y,fact  As double
  
  fact = r2d*4*60 'convert radians into seconds of time
  'Yearformula 1 - 7   : terms for EoT in seconds of time
  'Yearformula 8       : longitude of sun at 1 jan. 0h:0m:0s
  'Yearformula 9       : obliquity epsilon in degrees
  'Yearformula 10 - 13 : terms for decl. in degrees
  'Calculation of this formula is based on epoch 1900 but for Sundials
  'still valuable. 
  'Literature : 
  'Bulletin of De Zonnewijzerkring, nr. 8, march 1981 and nr. 22, february 1985,
  'articles by Thijs J. de Vries.

  d = CalcJulDay(xyear,1,0,0)-CalcJulDay(1900,1,0,12)
  l = 279.696678+0.9856473354*d+0.00002267*d*d/100000000.0
  l = l-Int(l/360)*360-360.0
  w = 281.220844+0.0000470684*d+0.00003390*d*d/100000000.0
  w = w-Int(w/360)*360
  e = 0.01675104-0.000011444*d/10000-0.0000000094*d*d/100000000.0
  epsilon= 23.452294-0.0035626*d/10000
  y = dTan(epsilon/2)*dTan(epsilon/2)
  yearformula(1) = (-2*e*dCos(w)-2*e*y*dCos(w))*fact
  yearformula(2) = ( 2*e*dSin(w)-2*e*y*dSin(w))*fact
  yearformula(3) = (y-5/4*e*e*dCos(2*w))*fact
  yearformula(4) = (  5/4*e*e*dSin(2*w))*fact
  yearformula(5) = ( 2*e*y*dCos(w))*fact
  yearformula(6) = (-2*e*y*dSin(w))*fact
  yearformula(7) = (-0.5*y*y)*fact
  yearformula(8) = l
  yearformula(9) = epsilon
  yearformula(10)=  2*e*dCos(w)*r2d
  yearformula(11)= -2*e*dSin(w)*r2d
  yearformula(12)=  5/4*e*e*dCos(2*w)*r2d
  yearformula(13)= -5/4*e*e*dSin(2*w)*r2d
End Sub


' --- Calculates EoT out of daynumber
Function EoT(ByVal daynr As double) As double

  Dim l,help,fact As Double
  'fact = 1          'use this fact for output in seconds of time
  'fact = 1/60       'use this fact for output in minutes of time
  fact = 1/60/4     'use this fact for output in degrees
  'fact = 1/60/4*d2r 'use this fact for output in radians
 
  l=daynr*360/365.2422+yearformula(8)
  help=yearformula(1)*dSin(l)+yearformula(2)*dCos(l)+yearformula(3)*dSin(2*l)
  help=help+yearformula(4)*dCos(2*l)+yearformula(5)*dSin(3*l)
  EoT=help+yearformula(6)*dCos(3*l)+yearformula(7)*dSin(4*l)
  Eot = EoT * fact
End Function


' --- Calculates sun's declination out of daynumber
Function decl(ByVal daynr As double) As Double

  Dim l,lambda,fact As double
  fact = 1    'use this fact for output in degrees
  'fact = d2r  'use this fact for output in radians
  
  l=daynr*360/365.2422+yearformula(8)
  lambda=l+yearformula(10)*dSin(l)+yearformula(11)*dCos(l)
  lambda=lambda+yearformula(12)*dSin(2*l)+yearformula(13)*dCos(2*l)
  decl=dAsin(dSin(lambda)*dsin(yearformula(9)))
  decl = decl * fact  
End Function


' --- Calculates daynumber out of year, month, day, hour, minutes
Function daynr(ByVal y,M,d,h,mi As double) As Double

  Dim a,help As double

  a=Int((M+9.0)/12.0)
  help=Int(275.0*M/9.0)-2.0*a+d-30.0+h/24.0+mi/60.0/24.0
  If y/4 = Int(y/4) Then
    daynr=help+a
  Else
    daynr=help
  End If
End Function

' --- Calculates half day arc out of Latitude and declination
Function halfdayarc(ByVal lat As double, ByVal decl As double) As double
  Const maxval = 89.9999
  Dim help,fact As Double
  fact = 1    'use this fact for output in degrees
  'fact = d2r  'use this fact for output in radians
  
If Abs(lat-decl) >= maxval Then
    halfdayarc = 0
  Else
    If Abs(lat+decl) >= maxval Then
      halfdayarc = 180
    Else
      help = -dTan(lat)*dTan(decl)
      halfdayarc = dAcos(help) * fact
    End If
  End If 
End Function

'Exclusive routines for this Azimuthal nomograph

Sub InitConstants
Scx = 1                'Scaling factors
Scy = 1
ScaleWidth  = 5      'Default length for half azimuth scale in inches in x direction
ScaleHeight = 3.3333 'Default length for half date    scale in inches in y direction

ScaleFont   = 0.047 '0.045
'Fonts are not changed by changing Scalewidth, Scaleheight, Scx or Scy.
'Choose scale for fonts as separate value. 

Scalex = ScaleWidth  * Scx  'Used in calculations
Scaley = ScaleHeight * Scy

tickmark1 = 0.05 * Scalex
tickmark2 = 0.025 * Scalex

deltax = 1 * Scx'0.15 * Scalex     'space between datescale and azmuth scale
deltay = 1 * Scy'0.30 * Scaley
     
TextX =  scalex + deltax - tickmark1 -0.066666 * Scx 
TextY =  scaley + deltay - tickmark1 -0.066666 * Scy
 
Xbox = 1.1 * Scalex       'Add 10% to Scalex for box around drawing
Ybox = 1.1 * Scaley       'Add 10% to Scaley for box around drawing

datetext(1) = "JAN"
datetext(2) = "FEV"
datetext(3) = "MAR"
datetext(4) = "ABR"
datetext(5) = "MAI"
datetext(6) = "JUN"
datetext(7) = "JUL"
datetext(8) = "AGO"
datetext(9) = "SET"
datetext(10) = "OUT"
datetext(11) = "NOV"
datetext(12) = "DEZ"

NumDaysInMonth(1) = 31
NumDaysInMonth(2) = 28
NumDaysInMonth(3) = 31
NumDaysInMonth(4) = 30
NumDaysInMonth(5) = 31
NumDaysInMonth(6) = 30
NumDaysInMonth(7) = 31
NumDaysInMonth(8) = 31
NumDaysInMonth(9) = 30
NumDaysInMonth(10) = 31
NumDaysInMonth(11) = 30
numdaysinmonth(12) = 31
End Sub


Sub InputData
Begin Dialog CONSTANTS_INPUT 13,1, 200, 160, "Digite dados do relgio de sol"
  Text 62,0,169,10, "RELGIO DE SOL AZIMUTAL"
  Text 32,20,100,10, "Latitude  (neg. Sul)"
  Text 32,32,100,10, "Meridiano Local (neg. Leste)"
  Text 32,44,130,10, "Zona Meridiano (neg. Leste)"
  Text 32,68,130,10, "EdT: 0 = sem   1 = com"
  Text 32,80,130,10, "Ms incio: 1 = Janeiro 7 = Julho"
  TextBox 132,20,37,10, .lat
  TextBox 132,32,37,10, .local_mer
  TextBox 132,44,37,10, .standard_mer
  TextBox 149,68,20,10, .EoTfact
  TextBox 149,80,20,10, .MonthFact
  OKButton 84,96,37,12
  Text 32,116,200,10, "Linhas horrias Italianas e Babilnicas so"
  Text 32,128,200,10, "em camadas   It-...  e   Bab-..."
  Text 32, 140,200,10,"Alternar camadas: ligadas ou desligadas."
End Dialog

'Initialize 
Dim prompt As constants_input
prompt.lat = 52
prompt.local_mer = -5
prompt.standard_mer = -15
prompt.EoTfact = 1
prompt.MonthFact = 1
repeat_until_inputcorrect: 'label to return to if input is not correct
action = Dialog(prompt)    'get the input

'Test the input
If (prompt.lat >-23.44 And prompt.lat <23.44)  = true Then 
  outtext = "Latitude" & " no pode estar entre -23.44 and +23.44"
  MsgBox outtext 
  GoTo repeat_until_inputcorrect
End If

If test("Latitude",prompt.lat,-66.56,66.56) = false Then 
  GoTo repeat_until_inputcorrect
End If

If test("local_meridian",prompt.local_mer,-180,180) = false Then
  GoTo repeat_until_inputcorrect
End If

If test("standard_meridian",prompt.standard_mer,-180,180) = false Then
  GoTo repeat_until_inputcorrect
End If

If prompt.EoTfact = 9 Then Stop
If test("EoTfact",prompt.EoTfact,0,1) = false Then
  GoTo repeat_until_inputcorrect
End If


If (prompt.MonthFact <> 1 And prompt.MonthFact <> 7) = true Then
  outtext = "Month is not 1 or 7"
  MsgBox outtext 
  GoTo repeat_until_inputcorrect
End If   

'Set program variables with input variables, angles in degrees
Latitude  = prompt.lat
LocalMeridian = prompt.local_mer
StandardMeridian = prompt.standard_mer
EoTfact = Int(prompt.EoTfact)
MonthFact = Int(prompt.MonthFact)
If Latitude < 1 Then hemi = 1 Else hemi = -1
End Sub


Function  test(varname,x,minval,maxval) As boolean
Dim outtext As String
If IsNumeric(x) = false  Then
test = false
outtext = varname & " dever ser numrica"
MsgBox outtext
exit Function
End If
If x < minval Or x > maxval Then
outtext = varname & " dever ser de " & chr$(13) & minval & "  at " & maxval
MsgBox outtext 
exit Function
End If
test = true
End Function


Sub DrawAzimuthDatescale
'Draw base of scales
dcSetCurrentLayer "DateAzScale"
x = scalex
y = scaley
dcSetLineParms dcBLACK, dcSOLID, dcTHIN
dcCreateLine -x, y + deltay, x , y + deltay
dcCreateLine -x, -y -deltay, x , -y -deltay
dcCreateLine -x - deltax, y, -x - deltax, -y
dcCreateLine x + deltax, -y, x + deltax, y

'Draw tickmarks for azimuth scale plus text.
'For Northern And Southern hemisphere the scale is different.
'Northern hemisphere: -180 through 0 to +180.
'Southern hemisphere:    0 through -/+180 to 0.
For count2 = -1 To 1 Step 2
If count2 = 1 Then dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 17,0,0
If count2 =-1 Then dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 25,0,0
For count1 = -180 To 180 Step 5
x1 = count1 * scalex / 180
y1 = count2 * (scaley + deltay)
x2 = x1
If count1 / 10 = Int(count1 / 10) Then
y2 = y1 - tickmark1 * count2
Else
y2 = y1 - tickmark2 * count2
End If
dcCreateLine x1, y1, x2, y2

If count1 / 30 = Int(count1 /30 ) Then
If Latitude > 0 Then temp2 = count1 Else temp2 = (180 - sgn(count1)*count1) * sgn(count1)
If Latitude < 0 And count1 = 0 Then temp2 = 180 
dcCreateText x2*hemi, TextY*count2, 0, temp2
End If

Next count1
Next count2

'Draw tickmarks for datescale plus text
'Tickmarks are placed for 1, 11 and 21 of the month.
'Last tickmark of december is first of january Next Year.
dcSetTextParms dcBlack,"Courier", "Normal", 90, 300 * ScaleFont, 22,0,0                                      
For count2 = -1 To 1 Step 2
If count2 = 1 Then dcSetTextParms dcBlack,"Courier", "Normal", 270, 300 * ScaleFont, 17,0,0
If count2 =-1 Then dcSetTextParms dcBlack,"Courier", "Normal", 270, 300 * ScaleFont, 25,0,0
sum1 = 0
If MonthFact = 1 Then
  temp1 = 1
  temp2 = 12
Else
  temp1 = 7
  temp2 = 18
End If
For count1 = temp1 To temp2                                 
x1 = (scalex + deltax) * count2
y1 = scaley - sum1 * scaley / 365 * 2   
x2 = x1 - tickmark1 * count2
y2 = y1
dcCreateLine x1, y1, x2, y2
If count2 = 1 Then
  DrawGrid1
End If
sum2 = sum1 + 10
x1 = (scalex + deltax) * count2
y1 = scaley - sum2 * scaley / 365 * 2
x2 = x1 - tickmark2 * count2
y2 = y1
dcCreateLine x1, y1, x2, y2
If count2 = 1 Then
  DrawGrid2
End If
If MonthFact = 1 Then 
  dcCreateText TextX*count2,y1-10*scaley / 365,0,datetext(count1)
Else
  count3 = count1
  If count1 > 12 Then count3 = count3 - 12
  dcCreateText TextX*count2,y1-10*scaley / 365,0,datetext(count3)
End If
sum2 = sum1 + 20
x1 = (scalex + deltax) * count2
y1 = scaley - sum2 * scaley / 365 * 2
x2 = x1 - tickmark2 * count2
y2 = y1
dcCreateLine x1, y1, x2, y2
If count2 = 1 Then
  DrawGrid2
End If
If MonthFact = 1 Then
  sum1 = sum1 + NumDaysInMonth(count1)
Else
  sum1 = sum1 + NumDaysInMonth(count3)
End If
x1 = (scalex + deltax) * count2
y1 = scaley - sum1 * scaley / 365 * 2
x2 = x1 - tickmark1 * count2
y2 = y1
dcCreateLine x1, y1, x2, y2
If count2 = 1 Then
  DrawGrid1
End If
Next count1
Next count2
End Sub

Sub DrawGrid1                  '1st of month
dcSetCurrentLayer "Grid1"
x1 = scalex
x2 = -x1
y2 = y1
dcCreateLine x1, y1, x2, y2
dcSetCurrentLayer "DateAzScale"
End Sub

Sub DrawGrid2                  '11th and 21st of month
dcSetCurrentLayer "Grid2"
dcSetLineParms dcBLACK, dcSTITCH, dcTHIN
x1 = scalex
x2 = -x1
y2 = y1
dcCreateLine x1, y1, x2, y2
dcSetLineParms dcBLACK, dcSOLID, dcTHIN
dcSetCurrentLayer "DateAzScale"
End Sub

Sub calculate_azimuth
Dim cotaz As Double
If hourangle = 0 Then hourangle = 0.0000001
If Abs(hourangle) = 180 Then hourangle = sgn(hourangle)*179.999999
cotaz = (dSin(Latitude)*dCos(hourangle)-dCos(Latitude)*dTan(declination))/dSin(hourangle)
azimuth = 90 - datan(cotaz)
If hourangle < 0 Then azimuth = azimuth - 180
End Sub


Sub DrawHourLines
For Hour = 0 To 23.75 Step 0.25
If Hour * 20 = Int(Hour * 20)  Then
dcSetCurrentLayer "Normal-Quarter-Hours"
dcSetLineParms dcBLACK, dcSOLID, dcTHIN
End If
If Hour * 10 = Int(Hour * 10) Then
dcSetCurrentLayer "Normal-Half-Hours"
dcSetLineParms dcBLUE, dcSOLID, dcTHIN
End If
If Hour = Int(Hour) Then
dcSetCurrentLayer "Normal-Full-Hours"
dcSetLineParms dcRED, dcSOLID, dcTHICK
End If
If Hour = 12 Then
dcSetCurrentLayer "Normal-Full-Hours"
dcSetLineParms dcPURPLE, dcSOLID, dcTHICK
End If
LongCorr = StandardMeridian - LocalMeridian
index = 0
index1 = 0
index2 = 0
linenumber = 0
flag = 1
If MonthFact = 1 Then
  temp1 = 0
  temp2 = 366
Else
  temp1 = 182
  temp2 = 548
End If
For count2 = temp1 To temp2 Step 1.5
If MonthFact = 1 Then
  count1 = count2 + 1
  If count1 > 366 Then count1 = 366
Else
count1 = count2
If count1 = 548 Then count1 = 547
End If
declination = decl(count1)
If EoTfact= 1 Then EqOfTime = EoT(count1) Else EqOfTime = 0
Hourangle = (Hour - 12) * 15 + LongCorr + EqOfTime
HalfDayLength = halfdayarc(Latitude, declination)
If Abs(hourangle) > 180 Then hourangle = (Abs(hourangle) - 360)*sgn(hourangle)
calculate_azimuth
If Latitude < 0 Then Azimuth = (180 - sgn(Azimuth)*Azimuth) * sgn(Azimuth)
x1 = Azimuth/180*scalex* hemi
If MonthFact = 1 Then
y1 = scaley - (count1-1)*(2*scaley/365)
Else
y1 = scaley - (count1-1 - 181)*(2*scaley/365)
End If
If Abs(Hourangle) <= HalfDayLength Then flag = 2
If Abs(Hourangle) > HalfDayLength Then flag = 1

'Routines to split a spline
If linenumber = 0 Then
If flag = 1 Then 
  GoTo xxxx:
End If
If flag = 2 Then
  linenumber = 1
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index1 = index1 + 2
  index = index + 2
End If
End If

If linenumber = 1 Then
If flag = 1 Then
  linenumber = 2 
  GoTo xxxx:
End If
If flag = 2 Then
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index1 = index1 + 2
  index = index + 2
End If
End If

If linenumber = 2 Then
If flag = 1 Then 
  GoTo xxxx:
End If
If flag = 2 Then
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index2 = index2 + 2
  index = index + 2
End If
End If

xxxx:
Next count2

'##############################################
'Use this for hourlines as SPLINES

'If index1 / 2 > 2 Then 
'dcCreateSpline lijn(1), index1 / 2, false
'If Hour = Int(Hour) Then                    'AddNumbersForHours
'dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 29,0,0
'x1 = lijn(1)
'y1 = scaley 
'dcCreateText x1, y1, 0, Hour
'dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 17,0,0
'dcCreateText x1, -y1 -0.07, 0, Hour + 1
'End If
'End If

'If index2 / 2 > 2 Then 
'dcCreateSpline lijn(index1 + 1), index2 / 2, false
'End If

'##############################################
'Use this for hourlines as LINES
If index1 / 2 > 2 Then
flag = 1
For count = 1 To index1 Step 2
If Hour = Int(Hour) And flag = 1 Then                    'AddNumbersForHours
dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 29,0,0
x1 = lijn(1)
y1 = scaley 
dcCreateText x1, y1, 0, Hour
dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 17,0,0
dcCreateText x1, -y1 -0.07, 0, Hour + 1
End If
x1 = lijn(count)
y1 = lijn(count + 1)
If flag = 2 Then 
dcCreateLine x2,y2,x1,y1  'tekenlijn
End If
x2 = x1
y2 = y1'
flag = 2
Next count
End If 

If index2 / 2 > 2 Then
flag = 1
For count = index1+1 To index Step 2
x1 = lijn(count)
y1 = lijn(count + 1)
If flag = 2 Then 
dcCreateLine x2,y2,x1,y1  'tekenlijn
End If
x2 = x1
y2 = y1
flag = 2
Next count
End If 

'End choice for hourlines as SPLINES or LINES
'##############################################

Next Hour

End Sub


Sub DrawSunriseSunsetLines
dcSetCurrentLayer "SunSetRise"
dcSetLineParms dcBLACK, dcSOLID, dcNORMAL
For count3 = -1 To 1 Step 2
index = 0
flag = 1
If MonthFact = 1 Then
  temp1 = 0
  temp2 = 366
Else
  temp1 = 182
  temp2 = 548
End If

For count2 = temp1 To temp2 Step 1.5

If MonthFact = 1 Then
count1 = count2 + 1
If count1 > 366 Then count1 = 366
Else
count1 = count2
If count1 = 548 Then count1 = 547
End If

declination = decl(count1)
Hourangle = halfdayarc(Latitude, declination)
calculate_azimuth
If Latitude < 0 Then Azimuth = (180 - sgn(Azimuth)*Azimuth) * sgn(Azimuth)
x1 = Azimuth/180*scalex * count3
If MonthFact = 1 Then
y1 = scaley - (count1-1)*(2*scaley/365)
Else
y1 = scaley - (count1-1 - 181)*(2*scaley/365)
End If

lijn(index + 1) = x1
lijn(index + 2) = y1
index = index + 2
yyyy:
flag = 2
Next count2

'##############################################
'Use this for sunrise / sunset lines as SPLINES

'dcCreateSpline lijn(1), index / 2, false

'##############################################
'Use this for sunrise / sunset lines as LINES

flag = 1
For count = 1 To index Step 2
x1 = lijn(count)
y1 = lijn(count + 1)
If flag = 2 Then 
dcCreateLine x2,y2,x1,y1  'tekenlijn
End If
x2 = x1
y2 = y1
flag = 2
Next count
 
'End choice for sunrise / sunset lines as SPLINES or LINES
'##############################################

Next count3
dcSetLineParms dcBLACK, dcSOLID, dcTHIN
End Sub


Sub DrawItBabHourLines

For ChoiceBabIt = -1 To 1 Step 2

For Hour = 0.25 To 23.75 Step 0.25

If ChoiceBabIt = -1 Then
  If Hour * 20 = Int(Hour * 20)  Then
  dcSetCurrentLayer "Bab-Quarter-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHIN
  End If
  If Hour * 10 = Int(Hour * 10) Then
  dcSetCurrentLayer "Bab-Half-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHIN
  End If
  If Hour = Int(Hour) Then
  dcSetCurrentLayer "Bab-Full-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHICK
  End If
End If

If ChoiceBabIt = 1 Then
  If Hour * 20 = Int(Hour * 20)  Then
  dcSetCurrentLayer "It-Quarter-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHIN
  End If
  If Hour * 10 = Int(Hour * 10) Then
  dcSetCurrentLayer "It-Half-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHIN
  End If
  If Hour = Int(Hour) Then
  dcSetCurrentLayer "It-Full-Hours"
  dcSetLineParms dcBLACK, dcSOLID, dcTHICK
  End If
End If


index = 0
index1 = 0
index2 = 0
linenumber = 0
flag = 1
If MonthFact = 1 Then
  temp1 = 0
  temp2 = 366
Else
  temp1 = 182
  temp2 = 548
End If
For count2 = temp1 To temp2 Step 1.5
If MonthFact = 1 Then
  count1 = count2 + 1
  If count1 > 366 Then count1 = 366
Else
count1 = count2
If count1 = 548 Then count1 = 547
End If
declination = decl(count1)
If EoTfact= 1 Then EqOfTime = EoT(count1) Else EqOfTime = 0
HalfDayLength = halfdayarc(Latitude, declination)  'No EoT or Longitude correction
Hourangle = (HalfDayLength - Hour * 15) * ChoiceBabIt
If Abs(hourangle) > 180 Then hourangle = (Abs(hourangle) - 360)*sgn(hourangle)
calculate_azimuth
If Latitude < 0 Then Azimuth = (180 - sgn(Azimuth)*Azimuth) * sgn(Azimuth)
x1 = Azimuth/180*scalex* hemi
If MonthFact = 1 Then
y1 = scaley - (count1-1)*(2*scaley/365)
Else
y1 = scaley - (count1-1 - 181)*(2*scaley/365)
End If
If Abs(Hourangle) <= HalfDayLength Then flag = 2
If Abs(Hourangle) > HalfDayLength Then flag = 1

'Routines to split a spline
If linenumber = 0 Then
If flag = 1 Then 
  GoTo xxxx:
End If
If flag = 2 Then
  linenumber = 1
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index1 = index1 + 2
  index = index + 2
End If
End If

If linenumber = 1 Then
If flag = 1 Then
  linenumber = 2 
  GoTo xxxx:
End If
If flag = 2 Then
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index1 = index1 + 2
  index = index + 2
End If
End If

If linenumber = 2 Then
If flag = 1 Then 
  GoTo xxxx:
End If
If flag = 2 Then
  lijn(index + 1) = x1
  lijn(index + 2) = y1
  index2 = index2 + 2
  index = index + 2
End If
End If

xxxx:
Next count2

'##############################################
'Use this for hourlines as SPLINES

'If index1 / 2 > 2 Then 
'dcCreateSpline lijn(1), index1 / 2, false
'If Hour = Int(Hour) Then                    'AddNumbersForHours
'dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 29,0,0
'x1 = lijn(1)
'y1 = scaley 
'dcCreateText x1, y1, 0, Hour
'dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 17,0,0
'dcCreateText x1, -y1 -0.07, 0, Hour + 1
'End If
'End If

'If index2 / 2 > 2 Then 
'dcCreateSpline lijn(index1 + 1), index2 / 2, false
'End If

'##############################################
'Use this for hourlines as LINES
If index1 / 2 > 2 Then
flag = 1
For count = 1 To index1 Step 2
If Hour = Int(Hour) And flag = 1 Then                    'AddNumbersForHours
dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 29,0,0
x1 = lijn(1)
y1 = scaley 
dcCreateText x1, y1, 0, Hour
dcSetTextParms dcBlack,"Courier", "Normal", 0, 400 * ScaleFont, 17,0,0
dcCreateText x1, -y1 -0.07, 0, Hour
End If
x1 = lijn(count)
y1 = lijn(count + 1)
If flag = 2 Then 
dcCreateLine x2,y2,x1,y1  'tekenlijn
End If
x2 = x1
y2 = y1'
flag = 2
Next count
End If 

If index2 / 2 > 2 Then
flag = 1
For count = index1+1 To index Step 2
x1 = lijn(count)
y1 = lijn(count + 1)
If flag = 2 Then 
dcCreateLine x2,y2,x1,y1  'tekenlijn
End If
x2 = x1
y2 = y1
flag = 2
Next count
End If 

'End choice for hourlines as SPLINES or LINES
'##############################################

Next Hour

Next ChoiceBabIt

End Sub 'DrawItBabHourLines


Sub AddConstants
'Add values for latitude and longitude in layer default
dcSetCurrentLayer "default"

outtext = "LATITUDE " & Latitude
dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 16,0,0
dcCreateText -scalex, scaley + .45, 0, outtext

dcSetCurrentLayer "Normal-Full-Hours"
outtext = "HORRIO DE VERO"
dcCreateText -scalex, -scaley - .30, 0, outtext

dcSetCurrentLayer "default"
outtext = "LONGITUDE " & localmeridian
dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 1,0,0
dcCreateText 0, scaley + .45, 0, outtext

outtext = "ZONA DO MERIDIANO" & standardmeridian
dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 34,0,0
dcCreateText scalex, scaley + .45, 0, outtext

dcSetCurrentLayer "Normal-Full-Hours"
outtext = "HORRIO DE VERO"
dcCreateText scalex, -scaley - .30, 0, outtext

dcSetCurrentLayer "It-Full-Hours"
outtext = "HORAS ITALIANAS"
dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 1,0,0
dcCreateText 0, -scaley - .30, 0, outtext

dcSetCurrentLayer "Bab-Full-Hours"
outtext = "HORAS BABILNICAS"
dcSetTextParms dcBlack,"Courier", "Normal", 0, 300 * ScaleFont, 1,0,0
dcCreateText 0, -scaley - .30, 0, outtext

dcSetCurrentLayer "default"
dcSetLineParms dcBLACK, dcSOLID, dcTHIN

End Sub


Sub SetLayers
dcAddLayer "DateAzScale"
dcAddLayer "Normal-Full-Hours"
dcAddLayer "Normal-Half-Hours"
dcAddLayer "Normal-Quarter-Hours"
dcAddLayer "SunSetRise"
dcAddLayer "Grid1"
dcAddLayer "Grid2"
dcAddLayer "It-Full-Hours"      ' = Italian Hours
dcAddLayer "It-Half-Hours"      ' = Italian Hours
dcAddLayer "It-Quarter-Hours"   ' = Italian Hours
dcAddLayer "Bab-Full-Hours"     ' = Babylonian Hours
dcAddLayer "Bab-Half-Hours"     ' = Babylonian Hours
dcAddLayer "Bab-Quarter-Hours"  ' = Babylonian Hours

End Sub


Sub DrawBoxAroundDrawing
'Draw box around drawing
dcSetCurrentLayer "default"
x1 = Xbox + deltax     
y1 = Ybox + deltay
x2 = - x1
y2 = - y1
dcCreateBox x1,y1,x2,y2
End Sub


'*********************